<?php
/**
 * سیستم ساخت اتوماتیک سرویس
 * ساخت اکانت VPN برای کاربران پس از خرید
 */

/**
 * ساخت اکانت VIP برای کاربر
 * 
 * @param int $user_id شناسه کاربر
 * @param int $plan_id شناسه تعرفه
 * @param string $hash_id شناسه پرداخت
 * @return array نتیجه عملیات
 */
function createServiceAccount($user_id, $plan_id, $hash_id){
    global $connection, $admin;
    
    try {
        // دریافت اطلاعات تعرفه
        $stmt = $connection->prepare("SELECT sp.*, s.server_ip, s.name as server_name
                                       FROM `service_plans` sp 
                                       JOIN `servers` s ON sp.server_id = s.id 
                                       WHERE sp.id = ? AND sp.status = 'active' AND s.status = 'active'");
        $stmt->bind_param("i", $plan_id);
        $stmt->execute();
        $plan = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if(!$plan){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'تعرفه یا سرور فعال نیست');
            return ['success' => false, 'error' => '⚠️ تعرفه یا سرور فعال نیست'];
        }
        
        // دریافت اطلاعات کاربر
        $stmt = $connection->prepare("SELECT * FROM `users` WHERE `userid` = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $user = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        if(!$user){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'کاربر یافت نشد');
            return ['success' => false, 'error' => '⚠️ کاربر یافت نشد'];
        }
        
        // اطلاعات API (همون که دیروز با هم ساختید)
        $api_url = 'http://' . $plan['server_ip'] . '/api/create_vip_account.php';
        $api_key = 'apn_vip_2024_secure_key_xyz123';
        
        // آماده‌سازی داده برای API
        // محدود کردن پلن‌های نامحدود به 200 گیگ
        $volume_to_send = $plan['volume_gb'];
        if($plan['volume_gb'] == 0 || $plan['volume_gb'] > 200){
            $volume_to_send = 200;
        }
        
        $api_data = [
            'action' => 'create_account',
            'user_id' => $user_id,
            'plan_id' => $plan_id,
            'volume_gb' => $volume_to_send,
            'days' => $plan['days'],
            'limit_ip' => 2
        ];
        
        // ارسال درخواست به API
        $ch = curl_init($api_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($api_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'X-API-Key: ' . $api_key
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        // بررسی خطای cURL
        if($curl_error){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'cURL error: ' . $curl_error);
            return ['success' => false, 'error' => '❌ خطا در اتصال به سرور: ' . $curl_error];
        }
        
        // بررسی کد HTTP
        if($http_code != 200){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'HTTP code: ' . $http_code);
            return ['success' => false, 'error' => '❌ خطای سرور (کد ' . $http_code . ')'];
        }
        
        // تبدیل پاسخ JSON
        $result = json_decode($response, true);
        
        if(!$result || !isset($result['success'])){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'Invalid response: ' . $response);
            return ['success' => false, 'error' => '❌ پاسخ نامعتبر از سرور'];
        }
        
        if(!$result['success']){
            $error = $result['error'] ?? 'خطای ناشناخته';
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'API error: ' . $error);
            return ['success' => false, 'error' => '❌ ' . $error];
        }
        
        // استخراج اطلاعات از پاسخ API
        $data = $result['data'];
        $email = $data['email'];
        $uuid = $data['uuid'];
        $config_link = $data['vless_link'];
        $inbound_id = $data['inbound_id'];
        
        // محاسبه تاریخ انقضا
        $expire_timestamp = time() + ($plan['days'] * 24 * 60 * 60);
        $expire_date = date('Y-m-d H:i:s', $expire_timestamp);
        
        // ذخیره در دیتابیس
        $stmt = $connection->prepare("INSERT INTO `service_orders` 
                                      (`user_id`, `plan_id`, `server_id`, `hash_id`, `email`, `uuid`, `inbound_id`, `config_link`, `volume_gb`, `days`, `price`, `expire_date`, `status`) 
                                      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active')");
        $stmt->bind_param("iiisssissiis", 
            $user_id, 
            $plan_id, 
            $plan['server_id'], 
            $hash_id, 
            $email, 
            $uuid, 
            $inbound_id, 
            $config_link, 
            $plan['volume_gb'], 
            $plan['days'], 
            $plan['price'], 
            $expire_date
        );
        
        if(!$stmt->execute()){
            logServiceCreation($user_id, $plan_id, $hash_id, 'failed', 'Database error: ' . $stmt->error);
            return ['success' => false, 'error' => '❌ خطا در ذخیره سرویس در دیتابیس'];
        }
        
        $order_id = $connection->insert_id;
        $stmt->close();
        
        // دریافت اطلاعات کاربر برای ارسال پیام
        $stmt = $connection->prepare("SELECT * FROM `users` WHERE `userid` = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $user = $stmt->get_result()->fetch_assoc();
        $stmt->close();
        
        // ثبت موفقیت در لاگ
        logServiceCreation($user_id, $plan_id, $hash_id, 'success', "Order ID: {$order_id}, UUID: {$uuid}, Email: {$email}");
        
        // ارسال کانفیگ به کاربر
        sendServiceToUser($user_id, $order_id, $plan, $config_link, $uuid);
        
        // اطلاع به ادمین
        if($user){
            notifyAdminAboutNewService($user, $plan, $order_id);
        }
        
        return [
            'success' => true,
            'order_id' => $order_id,
            'uuid' => $uuid,
            'config_link' => $config_link,
            'email' => $email
        ];
        
    } catch (Exception $e) {
        $error_msg = 'Exception: ' . $e->getMessage();
        logServiceCreation($user_id, $plan_id, $hash_id, 'failed', $error_msg);
        return ['success' => false, 'error' => '❌ خطای سیستمی: ' . $e->getMessage()];
    }
}

/**
 * ارسال کانفیگ به کاربر
 */
function sendServiceToUser($user_id, $order_id, $plan, $config_link, $uuid){
    $volume_text = $plan['volume_gb'] > 0 ? $plan['volume_gb'] . ' گیگابایت' : 'نامحدود';
    $price_text = number_format($plan['price']);
    $expire_date_shamsi = jdate('Y/m/d', strtotime('+' . $plan['days'] . ' days'));
    
    $message = "🎉 <b>تبریک! سرویس شما با موفقیت فعال شد</b>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━\n";
    $message .= "📦 <b>{$plan['name']}</b>\n";
    $message .= "━━━━━━━━━━━━━━━━━━\n\n";
    $message .= "📊 <b>مشخصات سرویس:</b>\n";
    $message .= "├ 🔋 حجم: <code>{$volume_text}</code>\n";
    $message .= "├ ⏰ مدت: <code>{$plan['days']} روز</code>\n";
    $message .= "├ 📅 تاریخ انقضا: <code>{$expire_date_shamsi}</code>\n";
    $message .= "└ 💰 مبلغ پرداختی: <code>{$price_text} تومان</code>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━\n";
    $message .= "🔑 <b>اطلاعات اتصال:</b>\n";
    $message .= "├ 🆔 شناسه سفارش: <code>#{$order_id}</code>\n";
    $message .= "└ 🔐 UUID: <code>{$uuid}</code>\n";
    $message .= "━━━━━━━━━━━━━━━━━━\n\n";
    $message .= "🔗 <b>لینک اتصال شما:</b>\n";
    $message .= "<code>{$config_link}</code>\n\n";
    $message .= "━━━━━━━━━━━━━━━━━━\n\n";
    $message .= "📱 <b>نحوه استفاده:</b>\n";
    $message .= "۱. لینک بالا را کپی کنید\n";
    $message .= "۲. در اپلیکیشن v2rayNG وارد کنید\n";
    $message .= "۳. روی Connect کلیک کنید\n\n";
    $message .= "❓ برای راهنمای کامل، دکمه زیر را بزنید.";
    
    $keyboard = [
        [['text' => '📱 راهنمای نصب و اتصال', 'callback_data' => 'connectionGuide']],
        [['text' => '📋 سرویس‌های من', 'callback_data' => 'myServices']],
        [['text' => '🏠 بازگشت به منو اصلی', 'callback_data' => 'mainMenu']]
    ];
    
    // استفاده از تابع bot() برای ارسال قابل اعتماد
    bot('sendMessage', [
        'chat_id' => $user_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * اطلاع به ادمین درباره سرویس جدید
 */
function notifyAdminAboutNewService($user, $plan, $order_id){
    global $admin;
    
    $volume_text = $plan['volume_gb'] > 0 ? $plan['volume_gb'] . ' گیگ' : 'نامحدود';
    $price_text = number_format($plan['price']);
    
    $message = "🔔 <b>سرویس جدید ایجاد شد</b>\n\n";
    $message .= "👤 کاربر: {$user['firstname']} (ID: {$user['userid']})\n";
    $message .= "📦 تعرفه: {$plan['name']}\n";
    $message .= "🔋 حجم: {$volume_text}\n";
    $message .= "⏰ مدت: {$plan['days']} روز\n";
    $message .= "💰 مبلغ: {$price_text} تومان\n";
    $message .= "🆔 شناسه سفارش: #{$order_id}";
    
    // استفاده از تابع bot() برای ارسال قابل اعتماد
    bot('sendMessage', [
        'chat_id' => $admin,
        'text' => $message,
        'parse_mode' => 'HTML'
    ]);
}

/**
 * ثبت لاگ ساخت سرویس
 */
function logServiceCreation($user_id, $plan_id, $hash_id, $status, $details){
    global $connection;
    
    $log_data = json_encode([
        'user_id' => $user_id,
        'plan_id' => $plan_id,
        'hash_id' => $hash_id,
        'status' => $status,
        'details' => $details,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
    // ثبت در فایل لاگ
    $log_file = __DIR__ . '/logs/service_creation_' . date('Y-m-d') . '.log';
    $log_dir = dirname($log_file);
    if(!is_dir($log_dir)){
        mkdir($log_dir, 0755, true);
    }
    
    file_put_contents($log_file, $log_data . "\n", FILE_APPEND);
}

/**
 * دریافت کانفیگ سرویس
 */
if(isset($data) && preg_match('/^getServiceConfig_(\d+)$/', $data, $match)){
    $order_id = $match[1];
    
    $stmt = $connection->prepare("SELECT so.*, sp.name as plan_name, sp.volume_gb, sp.days 
                                   FROM `service_orders` so 
                                   JOIN `service_plans` sp ON so.plan_id = sp.id 
                                   WHERE so.id = ? AND so.user_id = ?");
    $stmt->bind_param("ii", $order_id, $from_id);
    $stmt->execute();
    $service = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    if(!$service){
        alert("❌ سرویس یافت نشد!");
        exit;
    }
    
    // ارسال کانفیگ
    $config_text = "📥 <b>کانفیگ سرویس شما</b>\n\n";
    $config_text .= "🆔 شناسه: #{$order_id}\n";
    $config_text .= "📦 تعرفه: {$service['plan_name']}\n\n";
    $config_text .= "━━━━━━━━━━━━━━━━━━\n";
    $config_text .= "🔗 <b>لینک کانفیگ:</b>\n";
    $config_text .= "<code>{$service['config_link']}</code>\n";
    $config_text .= "━━━━━━━━━━━━━━━━━━\n\n";
    $config_text .= "📋 برای کپی روی کانفیگ کلیک کنید";
    
    delMessage();
    sendMessage($config_text, json_encode(['inline_keyboard' => [
        [['text' => '📱 راهنمای اتصال', 'callback_data' => 'connectionGuide']],
        [['text' => '🔙 بازگشت', 'callback_data' => 'myServices']]
    ]]), 'HTML');
    exit;
}

/**
 * نمایش لیست سرویس‌های کاربر
 */
if(isset($data) && $data == 'myServices'){
    $stmt = $connection->prepare("SELECT so.*, sp.name as plan_name, s.name as server_name 
                                   FROM `service_orders` so 
                                   JOIN `service_plans` sp ON so.plan_id = sp.id 
                                   JOIN `servers` s ON so.server_id = s.id 
                                   WHERE so.user_id = ? 
                                   ORDER BY so.id DESC LIMIT 10");
    $stmt->bind_param("i", $from_id);
    $stmt->execute();
    $services = $stmt->get_result();
    $stmt->close();
    
    $message = "📦 <b>سرویس‌های من</b>\n\n";
    
    if($services->num_rows > 0){
        while($service = $services->fetch_assoc()){
            $status_icon = $service['status'] == 'active' ? '✅' : ($service['status'] == 'expired' ? '⏰' : '❌');
            $volume_text = $service['volume_gb'] > 0 ? $service['volume_gb'] . ' گیگ' : 'نامحدود';
            $expire_shamsi = jdate('Y/m/d', strtotime($service['expire_date']));
            
            $message .= "━━━━━━━━━━━━━━━━━━\n";
            $message .= "{$status_icon} <b>{$service['plan_name']}</b>\n";
            $message .= "🖥️ سرور: {$service['server_name']}\n";
            $message .= "🔋 حجم: {$volume_text}\n";
            $message .= "📅 انقضا: {$expire_shamsi}\n";
            $message .= "🆔 #{$service['id']}\n";
            $message .= "━━━━━━━━━━━━━━━━━━\n\n";
        }
        
        $keyboard = [
            [['text' => '🔙 بازگشت', 'callback_data' => 'mainMenu']]
        ];
    } else {
        $message .= "📭 شما هنوز سرویسی خریداری نکرده‌اید.\n\n";
        
        $keyboard = [
            [['text' => '🛒 خرید سرویس', 'callback_data' => 'buyService']],
            [['text' => '🔙 بازگشت', 'callback_data' => 'mainMenu']]
        ];
    }
    
    editText($message_id, $message, json_encode(['inline_keyboard' => $keyboard]), 'HTML');
    exit;
}

/**
 * راهنمای اتصال
 */
if(isset($data) && $data == 'connectionGuide'){
    delMessage();
    
    $guide = "📱 <b>راهنمای اتصال به VPN</b>\n\n";
    $guide .= "۱️⃣ اپلیکیشن <b>v2rayNG</b> را از لینک‌های زیر دانلود کنید:\n\n";
    $guide .= "🤖 <a href='https://play.google.com/store/apps/details?id=com.v2ray.ang'>گوگل پلی</a>\n";
    $guide .= "🍎 <a href='https://apps.apple.com/us/app/v2box-v2ray-client/id6446814690'>اپ استور</a>\n\n";
    $guide .= "۲️⃣ اپلیکیشن را باز کنید\n\n";
    $guide .= "۳️⃣ روی آیکون ➕ کلیک کنید\n\n";
    $guide .= "۴️⃣ گزینه <b>Import config from Clipboard</b> را انتخاب کنید\n\n";
    $guide .= "۵️⃣ کانفیگ خود را کپی کرده و وارد کنید\n\n";
    $guide .= "۶️⃣ روی دکمه اتصال کلیک کنید ✅\n\n";
    $guide .= "━━━━━━━━━━━━━━━━━━\n";
    $guide .= "💡 در صورت بروز مشکل با پشتیبانی تماس بگیرید";
    
    sendMessage($guide, json_encode(['inline_keyboard' => [
        [['text' => '🔙 بازگشت', 'callback_data' => 'myServices']]
    ]]), 'HTML');
    exit;
}
?>
