<?php
/**
 * Auto Account Creator
 * سیستم ساخت خودکار اکانت برای پلن‌های VIP
 */

require_once 'config.php';
require_once 'vip_plans_handler.php';

/**
 * ساخت اکانت VIP بعد از تایید پرداخت
 */
function createVIPAccount($user_id, $plan_id, $hash_id) {
    global $connection, $vipPlans;
    
    // دریافت اطلاعات پلن
    if (!isset($vipPlans[$plan_id])) {
        logAutoAccount("❌ Plan not found: " . $plan_id);
        return [
            'success' => false,
            'message' => 'پلن مورد نظر یافت نشد!'
        ];
    }
    
    $plan = $vipPlans[$plan_id];
    $volume_gb = $plan['volume'];
    $days = $plan['days'];
    $limit_ip = $plan['users'];
    
    // ارسال درخواست به API روی VPS
    $result = callVPSApi('create_account', [
        'user_id' => $user_id,
        'plan_id' => $plan_id,
        'volume_gb' => $volume_gb,
        'days' => $days,
        'limit_ip' => $limit_ip
    ]);
    
    if (!$result || !$result['success']) {
        $error_msg = $result['error'] ?? 'خطای ناشناخته';
        logAutoAccount("❌ Failed to create account: " . $error_msg);
        return [
            'success' => false,
            'message' => '⚠️ خطا در ساخت اکانت: ' . $error_msg
        ];
    }
    
    $account_data = $result['data'];
    
    // ذخیره در دیتابیس
    $save_result = saveAccountToDB(
        $user_id,
        $hash_id,
        $plan_id,
        $account_data['email'],
        $account_data['uuid'],
        $account_data['inbound_id'],
        $volume_gb,
        $days,
        $account_data['vless_link']
    );
    
    if (!$save_result) {
        logAutoAccount("⚠️ Account created but failed to save to DB: " . $account_data['email']);
        return [
            'success' => false,
            'message' => '⚠️ اکانت ساخته شد اما در ذخیره‌سازی مشکل پیش آمد!'
        ];
    }
    
    logAutoAccount("✅ Account created successfully: " . $account_data['email']);
    
    return [
        'success' => true,
        'message' => '✅ اکانت شما با موفقیت ساخته شد!',
        'account' => [
            'email' => $account_data['email'],
            'uuid' => $account_data['uuid'],
            'volume_gb' => $volume_gb == 0 ? 'نامحدود' : $volume_gb . ' گیگ',
            'days' => $days . ' روز',
            'link' => $account_data['vless_link'],
            'expire_date' => $account_data['expire_date']
        ]
    ];
}

/**
 * فراخوانی API روی سرور VPS
 */
function callVPSApi($action, $data = []) {
    $data['action'] = $action;
    
    $ch = curl_init(VPS_API_URL);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'X-API-Key: ' . VPS_API_SECRET
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($error) {
        logAutoAccount("CURL Error: " . $error);
        return ['success' => false, 'error' => $error];
    }
    
    if ($http_code !== 200) {
        logAutoAccount("HTTP Error: " . $http_code);
        return ['success' => false, 'error' => "HTTP {$http_code}"];
    }
    
    $result = json_decode($response, true);
    
    if (!$result) {
        logAutoAccount("Invalid JSON response: " . $response);
        return ['success' => false, 'error' => 'Invalid response'];
    }
    
    return $result;
}

/**
 * تولید نام یونیک برای اکانت
 */
function generateAccountEmail($user_id, $plan_id) {
    $timestamp = time();
    $random = substr(md5(uniqid()), 0, 6);
    return "user_{$user_id}_{$plan_id}_{$random}";
}

/**
 * ذخیره اطلاعات اکانت در دیتابیس
 */
function saveAccountToDB($user_id, $hash_id, $plan_id, $email, $uuid, $inbound_id, $volume_gb, $days, $vless_link) {
    global $connection;
    
    $expire_date = time() + ($days * 86400);
    $created_at = time();
    $status = 'active';
    
    $stmt = $connection->prepare("
        INSERT INTO `vip_accounts` 
        (`user_id`, `hash_id`, `plan_id`, `email`, `uuid`, `inbound_id`, `volume_gb`, `days`, `vless_link`, `expire_date`, `created_at`, `status`) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    if (!$stmt) {
        logAutoAccount("❌ DB prepare failed: " . $connection->error);
        return false;
    }
    
    $stmt->bind_param(
        "issssiiiisis",
        $user_id,
        $hash_id,
        $plan_id,
        $email,
        $uuid,
        $inbound_id,
        $volume_gb,
        $days,
        $vless_link,
        $expire_date,
        $created_at,
        $status
    );
    
    $result = $stmt->execute();
    $stmt->close();
    
    return $result;
}

/**
 * دریافت اطلاعات اکانت از دیتابیس
 */
function getAccountByHashId($hash_id) {
    global $connection;
    
    $stmt = $connection->prepare("SELECT * FROM `vip_accounts` WHERE `hash_id` = ?");
    $stmt->bind_param("s", $hash_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    return $result;
}

/**
 * دریافت همه اکانت‌های یک کاربر
 */
function getUserAccounts($user_id) {
    global $connection;
    
    $stmt = $connection->prepare("SELECT * FROM `vip_accounts` WHERE `user_id` = ? ORDER BY `created_at` DESC");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $accounts = [];
    while ($row = $result->fetch_assoc()) {
        $accounts[] = $row;
    }
    
    $stmt->close();
    
    return $accounts;
}

/**
 * بررسی وضعیت اکانت از سرور
 */
function checkAccountStatus($email) {
    $xui = new XUI_API(XUI_PANEL_URL, XUI_USERNAME, XUI_PASSWORD, false);
    
    if (!$xui->login()) {
        return null;
    }
    
    $info = $xui->getClientInfo($email);
    
    if (!$info) {
        return null;
    }
    
    return [
        'email' => $info['email'] ?? $email,
        'upload' => isset($info['up']) ? formatBytes($info['up']) : '0',
        'download' => isset($info['down']) ? formatBytes($info['down']) : '0',
        'total' => isset($info['total']) ? formatBytes($info['total']) : '0',
        'enable' => isset($info['enable']) ? $info['enable'] : false,
        'expiry' => isset($info['expiryTime']) ? date('Y-m-d H:i', $info['expiryTime'] / 1000) : 'نامحدود'
    ];
}

/**
 * تمدید اکانت
 */
function renewVIPAccount($email, $inbound_id, $add_gb, $add_days) {
    $xui = new XUI_API(XUI_PANEL_URL, XUI_USERNAME, XUI_PASSWORD, XUI_DEBUG);
    
    if (!$xui->login()) {
        return false;
    }
    
    $result = $xui->renewClient($inbound_id, $email, $add_gb, $add_days);
    
    if ($result) {
        logAutoAccount("✅ Account renewed: " . $email . " (+{$add_gb}GB, +{$add_days}days)");
        return true;
    }
    
    logAutoAccount("❌ Failed to renew account: " . $email);
    return false;
}

/**
 * حذف اکانت
 */
function deleteVIPAccount($email, $inbound_id) {
    $xui = new XUI_API(XUI_PANEL_URL, XUI_USERNAME, XUI_PASSWORD, XUI_DEBUG);
    
    if (!$xui->login()) {
        return false;
    }
    
    $result = $xui->deleteClient($inbound_id, $email);
    
    if ($result) {
        logAutoAccount("✅ Account deleted: " . $email);
        
        // به‌روزرسانی وضعیت در دیتابیس
        global $connection;
        $stmt = $connection->prepare("UPDATE `vip_accounts` SET `status` = 'deleted' WHERE `email` = ?");
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $stmt->close();
        
        return true;
    }
    
    logAutoAccount("❌ Failed to delete account: " . $email);
    return false;
}

/**
 * فرمت کردن بایت به واحد قابل خواندن
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

/**
 * لاگ عملیات
 */
function logAutoAccount($message) {
    $log_file = __DIR__ . '/auto_account.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[{$timestamp}] {$message}\n", FILE_APPEND);
}

/**
 * ساخت پیام اکانت برای ارسال به کاربر
 */
function formatAccountMessage($account_data) {
    $message = "🎉 <b>اکانت VIP شما آماده است!</b>\n\n";
    $message .= "📧 <b>نام اکانت:</b> <code>{$account_data['account']['email']}</code>\n";
    $message .= "📦 <b>حجم:</b> {$account_data['account']['volume_gb']}\n";
    $message .= "⏰ <b>مدت زمان:</b> {$account_data['account']['days']}\n";
    $message .= "📅 <b>تاریخ انقضا:</b> {$account_data['account']['expire_date']}\n\n";
    
    if (!empty($account_data['account']['link'])) {
        $message .= "🔗 <b>لینک اتصال:</b>\n";
        $message .= "<code>{$account_data['account']['link']}</code>\n\n";
        $message .= "💡 <i>روی لینک بالا کلیک کنید تا کپی شود</i>\n\n";
    }
    
    $message .= "📱 <b>راهنمای اتصال:</b>\n";
    $message .= "1️⃣ نرم‌افزار V2Ray یا Shadowrocket را نصب کنید\n";
    $message .= "2️⃣ روی لینک بالا کلیک کنید\n";
    $message .= "3️⃣ اتصال را برقرار کنید\n\n";
    $message .= "✅ از خرید شما متشکریم!";
    
    return $message;
}
