# 📋 پیشنهادات بهبود UX و امنیت ربات

## 🎨 بهبودهای تجربه کاربری (UX)

### 1. صفحه اصلی و منوی اصلی
✅ **انجام شده:**
- دکمه "سرویس‌های من" اضافه شد
- دکمه "پلن‌های Trading" حذف شد

⚠️ **پیشنهادات:**
- **ترتیب دکمه‌ها:** دکمه‌های پرکاربردتر (خرید سرویس، سرویس‌های من) در ردیف اول قرار بگیرند
- **آیکون‌های بهتر:** 
  - 🛍️ خرید سرویس
  - 📱 سرویس‌های من
  - 💰 کیف پول من
  - 🎟️ کد تخفیف
  - 👤 اطلاعات من
  - 💬 پشتیبانی
- **پیام خوشامدگویی:** یک پیام خوشامدگویی جذاب با توضیح کوتاه درباره خدمات

### 2. فرآیند خرید سرویس
✅ **انجام شده:**
- نمایش "نامحدود (حداکثر 200GB)" برای پلن‌های نامحدود
- اضافه شدن دکمه کد تخفیف به صفحه پرداخت

⚠️ **پیشنهادات:**
- **مراحل واضح‌تر:**
  ```
  گام 1 از 4: انتخاب دسته‌بندی
  گام 2 از 4: انتخاب پلن
  گام 3 از 4: انتخاب روش پرداخت
  گام 4 از 4: تکمیل پرداخت
  ```
- **نمایش مقایسه‌ای:** یک دکمه "مقایسه پلن‌ها" برای دیدن تفاوت‌ها
- **پیش‌نمایش:**  
  - نمایش سرعت تقریبی
  - تعداد دستگاه‌های قابل اتصال
  - کیفیت سرویس (طلایی، نقره‌ای، برنز)
- **تأیید نهایی:** قبل از پرداخت، یک صفحه تأیید نهایی با خلاصه سفارش

### 3. کد تخفیف
✅ **انجام شده:**
- سیستم کامل کد تخفیف با ثبت استفاده
- نمایش قیمت با تخفیف

⚠️ **پیشنهادات:**
- **پیام خطا واضح‌تر:**
  ```
  ❌ کد تخفیف "WELCOME20" منقضی شده است
  📅 تاریخ انقضا: 1403/09/15
  ```
- **پیشنهاد کدهای جایگزین:** اگر کد منقضی شده، کدهای فعال دیگر پیشنهاد شود
- **نمایش کدهای عمومی:** یک دکمه "کدهای تخفیف فعال" برای دیدن کدهای عمومی

### 4. سیستم کیف پول
✅ **وضعیت فعلی:**
- افزایش موجودی با درگاه‌های مختلف
- حداقل شارژ 5000 تومان

⚠️ **پیشنهادات:**
- **نمایش تاریخچه تراکنش‌ها:**
  ```
  📊 تاریخچه کیف پول
  
  ✅ 1403/09/20 - شارژ +100,000 تومان
  ❌ 1403/09/19 - خرید سرویس -50,000 تومان
  ✅ 1403/09/15 - شارژ +200,000 تومان
  ```
- **پیشنهاد بسته‌های شارژ:**
  ```
  🎁 بسته 100 هزار تومان + 10% شارژ هدیه
  🎁 بسته 500 هزار تومان + 15% شارژ هدیه
  🎁 بسته 1 میلیون تومان + 20% شارژ هدیه
  ```
- **هشدار موجودی کم:** اگر موجودی کمتر از 10,000 تومان باشد، پیشنهاد شارژ

### 5. سرویس‌های من
✅ **انجام شده:**
- لیست سرویس‌ها با جزئیات
- دکمه دانلود کانفیگ

⚠️ **پیشنهادات:**
- **فیلتر و جستجو:**
  - فیلتر بر اساس وضعیت (فعال، منقضی، نزدیک به اتمام)
  - جستجو با شناسه سفارش
- **اطلاعیه‌های هوشمند:**
  ```
  ⚠️ سرویس #1234 شما 3 روز دیگر منقضی می‌شود
  📊 حجم مصرفی شما: 15GB از 50GB (30%)
  ```
- **تمدید سریع:** دکمه "تمدید" کنار هر سرویس
- **نمایش QR Code:** برای اسکن سریع در گوشی
- **راهنمای نصب:** لینک به راهنمای نصب برای سیستم‌عامل‌های مختلف

### 6. پشتیبانی
⚠️ **پیشنهادات:**
- **تیکتینگ ساده:**
  ```
  📋 تیکت‌های من
  
  🟢 #123 - در حال بررسی - "مشکل اتصال"
  🔴 #122 - بسته شده - "سوال درباره تمدید"
  
  ➕ تیکت جدید
  ```
- **سوالات متداول (FAQ):** قبل از ارسال تیکت، FAQ نمایش داده شود
- **پاسخ خودکار:** برای سوالات رایج مثل "چطور کانفیگ را نصب کنم؟"

### 7. اعلان‌ها
⚠️ **پیشنهادات:**
- **اعلان 3 روز قبل از انقضا:**
  ```
  ⏰ سرویس شما 3 روز دیگر منقضی می‌شود!
  
  📦 پلن: ماهانه 50 گیگ
  📅 تاریخ انقضا: 1403/09/25
  
  🔄 تمدید سریع
  ```
- **اعلان اتمام 80% حجم:**
  ```
  📊 شما 80% حجم خود را مصرف کرده‌اید
  
  مصرفی: 40GB از 50GB
  باقیمانده: 10GB
  
  🛒 خرید پلن جدید
  ```
- **اعلان پرداخت موفق:** فوری بعد از پرداخت با جزئیات

---

## 🔒 بهبودهای امنیتی

### 1. ورودی کاربر (Input Validation)
❌ **مشکلات فعلی:**
```php
// بدون اعتبارسنجی کامل
if($text == "/empty"){
```

✅ **راه‌حل:**
```php
// اعتبارسنجی کامل
function sanitizeInput($input) {
    $input = trim($input);
    $input = strip_tags($input);
    $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
    return $input;
}

$safe_text = sanitizeInput($text);
```

### 2. SQL Injection Prevention
✅ **وضعیت فعلی:** استفاده از Prepared Statements (خوب)
```php
$stmt = $connection->prepare("SELECT * FROM users WHERE userid = ?");
$stmt->bind_param("i", $user_id);
```

⚠️ **نکته مهم:** همیشه از Prepared Statements استفاده کنید، هرگز مستقیم متغیر در query نگذارید

### 3. امنیت API
❌ **مشکلات فعلی:**
```php
$api_key = 'apn_vip_2024_secure_key_xyz123'; // هاردکد شده
```

✅ **راه‌حل:**
- API Key در فایل `.env` یا خارج از public directory
- استفاده از Environment Variables
- تغییر دوره‌ای API Key (هر 3 ماه)
- Rate Limiting برای جلوگیری از Brute Force

```php
// config.php
define('API_KEY', getenv('VPS_API_KEY'));
define('API_SECRET', getenv('VPS_API_SECRET'));

// ثبت لاگ هر درخواست API
function logApiCall($endpoint, $user_id, $response_code) {
    error_log("API Call: {$endpoint} | User: {$user_id} | Code: {$response_code}");
}
```

### 4. امنیت فایل‌ها
❌ **مشکل:**
- فایل‌های حساس مثل `config.php` در دسترس عمومی

✅ **راه‌حل:**
```php
// در ابتدای config.php
if (!defined('SECURE_ACCESS')) {
    die('Direct access not permitted');
}

// در bot.php
define('SECURE_ACCESS', true);
require_once 'config.php';
```

### 5. محافظت از Session/Step
❌ **مشکل فعلی:**
```php
setUser($data); // بدون timeout
```

✅ **راه‌حل:**
```php
// اضافه کردن timeout برای step
function setUser($step = null, $field = 'step') {
    global $connection, $from_id;
    
    if($step !== null) {
        $timeout = time() + 600; // 10 دقیقه
        $stmt = $connection->prepare("UPDATE users SET {$field} = ?, step_timeout = ? WHERE userid = ?");
        $stmt->bind_param("sii", $step, $timeout, $from_id);
    } else {
        $stmt = $connection->prepare("UPDATE users SET {$field} = 'none', step_timeout = NULL WHERE userid = ?");
        $stmt->bind_param("i", $from_id);
    }
    $stmt->execute();
    $stmt->close();
}

// چک کردن timeout
if($userInfo['step_timeout'] && time() > $userInfo['step_timeout']) {
    setUser(); // ریست کردن step
    sendMessage("⏱️ زمان شما به پایان رسید. لطفا دوباره تلاش کنید.", getMainKeys());
    exit;
}
```

### 6. Rate Limiting
⚠️ **پیشنهاد:**
```php
// محدود کردن تعداد درخواست‌ها
function checkRateLimit($user_id, $action, $max_requests = 10, $window = 60) {
    global $connection;
    
    $stmt = $connection->prepare("SELECT COUNT(*) as cnt FROM rate_limits 
                                  WHERE user_id = ? AND action = ? AND created_at > ?");
    $time_limit = time() - $window;
    $stmt->bind_param("isi", $user_id, $action, $time_limit);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    
    if($result['cnt'] >= $max_requests) {
        return false;
    }
    
    // ثبت درخواست جدید
    $stmt = $connection->prepare("INSERT INTO rate_limits (user_id, action, created_at) VALUES (?, ?, ?)");
    $now = time();
    $stmt->bind_param("isi", $user_id, $action, $now);
    $stmt->execute();
    $stmt->close();
    
    return true;
}

// استفاده
if(!checkRateLimit($from_id, 'buy_service', 5, 60)) {
    alert("⚠️ شما بیش از حد مجاز درخواست داده‌اید. لطفا 1 دقیقه صبر کنید.");
    exit;
}
```

جدول مورد نیاز:
```sql
CREATE TABLE IF NOT EXISTS `rate_limits` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `user_id` BIGINT NOT NULL,
  `action` VARCHAR(50) NOT NULL,
  `created_at` INT NOT NULL,
  INDEX `idx_user_action` (`user_id`, `action`),
  INDEX `idx_created` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

### 7. لاگ‌گیری امنیتی
✅ **راه‌حل:**
```php
// ثبت رویدادهای مهم
function logSecurityEvent($event_type, $user_id, $details = '') {
    global $connection;
    
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    $stmt = $connection->prepare("INSERT INTO security_logs 
                                  (event_type, user_id, ip_address, user_agent, details, created_at) 
                                  VALUES (?, ?, ?, ?, ?, ?)");
    $now = time();
    $stmt->bind_param("sisssi", $event_type, $user_id, $ip, $user_agent, $details, $now);
    $stmt->execute();
    $stmt->close();
    
    // ثبت در error_log
    error_log("Security: {$event_type} | User: {$user_id} | IP: {$ip} | Details: {$details}");
}

// مثال استفاده
logSecurityEvent('PAYMENT_SUCCESS', $from_id, "Plan ID: {$plan_id}, Amount: {$final_price}");
logSecurityEvent('LOGIN_FAILED', $from_id, "Invalid code");
logSecurityEvent('DISCOUNT_USED', $from_id, "Code: {$discount['code']}");
```

### 8. بررسی دسترسی ادمین
✅ **وضعیت فعلی:**
```php
if($from_id == $admin || $userInfo['isAdmin'] == true)
```

⚠️ **بهبود:**
```php
// تابع مرکزی برای چک ادمین
function isAdmin($user_id, $user_info) {
    global $admin;
    
    // چک اصلی
    if($user_id == $admin) return true;
    
    // چک لیست ادمین‌ها از دیتابیس
    if(isset($user_info['isAdmin']) && $user_info['isAdmin'] === true) {
        return true;
    }
    
    return false;
}

// استفاده
if(!isAdmin($from_id, $userInfo)) {
    alert("⛔ شما دسترسی ندارید!");
    exit;
}
```

### 9. محافظت از Webhook
⚠️ **پیشنهاد:**
```php
// در ابتدای bot.php
function verifyTelegramRequest() {
    $secret_key = hash('sha256', BOT_TOKEN, true);
    
    if (!isset($_SERVER['HTTP_X_TELEGRAM_BOT_API_SECRET_TOKEN'])) {
        return false;
    }
    
    $received_hash = $_SERVER['HTTP_X_TELEGRAM_BOT_API_SECRET_TOKEN'];
    $expected_hash = hash_hmac('sha256', file_get_contents('php://input'), $secret_key);
    
    return hash_equals($expected_hash, $received_hash);
}

// چک کردن
if(!verifyTelegramRequest()) {
    http_response_code(403);
    die('Forbidden');
}
```

### 10. پشتیبان‌گیری خودکار
⚠️ **پیشنهاد:**
```bash
#!/bin/bash
# backup.sh - اجرا روزانه با cron

DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_DIR="/backup/robotali"
DB_NAME="lotfisch_qq"

# بکاپ دیتابیس
mysqldump -u username -p'password' $DB_NAME > $BACKUP_DIR/db_$DATE.sql

# بکاپ فایل‌ها
tar -czf $BACKUP_DIR/files_$DATE.tar.gz /path/to/robotali

# حذف بکاپ‌های قدیمی‌تر از 30 روز
find $BACKUP_DIR -name "*.sql" -mtime +30 -delete
find $BACKUP_DIR -name "*.tar.gz" -mtime +30 -delete

# crontab
# 0 3 * * * /path/to/backup.sh
```

---

## 📊 بهبودهای Performance

### 1. کش کردن
```php
// کش کردن پلن‌ها برای 5 دقیقه
function getCachedPlans() {
    $cache_file = '/tmp/plans_cache.json';
    $cache_time = 300; // 5 دقیقه
    
    if(file_exists($cache_file) && (time() - filemtime($cache_file) < $cache_time)) {
        return json_decode(file_get_contents($cache_file), true);
    }
    
    // دریافت از دیتابیس
    global $connection;
    $stmt = $connection->prepare("SELECT * FROM service_plans WHERE status = 'active'");
    $stmt->execute();
    $plans = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
    
    file_put_contents($cache_file, json_encode($plans));
    return $plans;
}
```

### 2. بهینه‌سازی Query
```php
// بد ❌
$stmt = $connection->prepare("SELECT * FROM service_orders WHERE user_id = ?");

// خوب ✅
$stmt = $connection->prepare("SELECT id, plan_id, status, expire_date FROM service_orders WHERE user_id = ? LIMIT 50");
```

### 3. Index دیتابیس
```sql
-- اضافه کردن index‌های مناسب
ALTER TABLE `service_orders` ADD INDEX `idx_user_status` (`user_id`, `status`);
ALTER TABLE `pays` ADD INDEX `idx_state_created` (`state`, `request_date`);
ALTER TABLE `discounts` ADD INDEX `idx_code_status` (`code`, `status`);
```

---

## ✅ چک‌لیست امنیتی نهایی

- [ ] همه API Key‌ها در محیط محرمانه
- [ ] همه ورودی‌های کاربر Sanitize می‌شوند
- [ ] فقط Prepared Statement برای query
- [ ] Rate Limiting فعال است
- [ ] لاگ امنیتی برای رویدادهای حساس
- [ ] بکاپ خودکار روزانه
- [ ] SSL برای وب‌سایت فعال است
- [ ] Webhook فقط از IP تلگرام درخواست می‌پذیرد
- [ ] Session Timeout برای step‌های کاربر
- [ ] پیام‌های خطا اطلاعات حساس فاش نمی‌کنند

---

## 📝 نتیجه‌گیری

این پیشنهادات به ترتیب اولویت:

**اولویت بالا (فوری):**
1. ✅ Rate Limiting
2. ✅ Input Validation
3. ✅ Security Logging
4. ✅ بکاپ خودکار

**اولویت متوسط:**
5. نمایش تاریخچه تراکنش‌ها
6. سیستم اعلان‌های هوشمند
7. تمدید سریع سرویس
8. بهینه‌سازی Query

**اولویت پایین:**
9. سیستم تیکتینگ
10. FAQ خودکار
11. کش کردن داده‌ها
12. مقایسه پلن‌ها
